/*
 * MazeGrid.java
 *
 * Created on March 2, 2003, 11:10 AM
 */

package nom.DannyBrewer.recreation.squareMaze;

import nom.DannyBrewer.utils.PropertyListNonPersistent;

/**
 * <p>
 * MazeGrid is the representation of a maze.
 * Methods are provided to set or get the states of the
 *  cells within the maze.
 * The maze structure remains internally consistent because it
 *  can only be altered through the use of the public api functions.
 * </p>
 *
 * @author  danny brewer
 */
public class SquareMaze {
	
    //----------------------------------------------------------------------
    //  Constructor
    //----------------------------------------------------------------------
	
	// Create a maze of a specified size.
	public SquareMaze(int rows, int cols) {
		allocateCells( rows, cols );
	}
	
    //----------------------------------------------------------------------
    //  Public API - directional
    //----------------------------------------------------------------------

	public static final int DIRECTION_TOP = 0;
	public static final int DIRECTION_LEFT = 1;
	public static final int DIRECTION_RIGHT = 2;
	public static final int DIRECTION_BOTTOM = 3;
	
	public static final int FIRST_DIRECTION = DIRECTION_TOP;
	public static final int LAST_DIRECTION = DIRECTION_BOTTOM;
	
	
	public static final String DRECTION_TOP_NAME = "Up";
	public static final String DRECTION_LEFT_NAME = "Left";
	public static final String DRECTION_RIGHT_NAME = "Right";
	public static final String DRECTION_BOTTOM_NAME = "Down";
	
	
	// Return the name of a direction.
	public static String directionName( int direction ) {
		switch( direction ) {
			case DIRECTION_TOP:		return DRECTION_TOP_NAME;
			case DIRECTION_LEFT:	return DRECTION_LEFT_NAME;
			case DIRECTION_RIGHT:	return DRECTION_RIGHT_NAME;
			//case DIRECTION_BOTTOM:
			default: return DRECTION_BOTTOM_NAME;
		}
	}

	
	// Return the opposite direction.
	public static int oppositeDirection( int direction ) {
		switch( direction ) {
			case DIRECTION_TOP:		return DIRECTION_BOTTOM;
			case DIRECTION_LEFT:	return DIRECTION_RIGHT;
			case DIRECTION_RIGHT:	return DIRECTION_LEFT;
			//case DIRECTION_BOTTOM:
			default: return DIRECTION_TOP;
		}
	}

	// If you enter a cell from a certian direction,
	//  and then make a right hand turn,
	//  this returns which direction would you be going.
	// If you want to make a right hand turn
	//  from the direction you entered a cell,
	//  this returns which direction you should exit the cell.
	// When exiting a cell, use the opposite direction,
	//  or call turnLeftDirection instead.
	public static int turnRightDirection( int direction ) {
		switch( direction ) {
			case DIRECTION_TOP:		return DIRECTION_LEFT;
			case DIRECTION_LEFT:	return DIRECTION_BOTTOM;
			case DIRECTION_RIGHT:	return DIRECTION_TOP;
			//case DIRECTION_BOTTOM:
			default: return DIRECTION_RIGHT;
		}
	}

	// If you enter a cell from a certian direction,
	//  and then make a left hand turn,
	//  this returns which direction would you be going.
	// If you want to make a left hand turn
	//  from the direction you entered a cell,
	//  this returns which direction you should exit the cell.
	// When exiting a cell, use the opposite direction,
	//  or call turnRightDirection instead.
	public static int turnLeftDirection( int direction ) {
		switch( direction ) {
			case DIRECTION_TOP:		return DIRECTION_RIGHT;
			case DIRECTION_LEFT:	return DIRECTION_TOP;
			case DIRECTION_RIGHT:	return DIRECTION_BOTTOM;
			//case DIRECTION_BOTTOM:
			default: return DIRECTION_LEFT;
		}
	}


    //----------------------------------------------------------------------
    //  Public API -- maze cells
    //----------------------------------------------------------------------

	// Allocate the maze to have a certian initial size.
	// All cells are in the inital state.
	// Each cell has four walls and is unreachable.
	public void allocateCells( int rows, int cols ) {
		this.rows = rows;
		this.cols = cols;
		
		// Allocate the cells
		mazeCells = new MazeCell[rows][];
		for( int row = 0; row < rows; ++row ) {
			mazeCells[row] = new MazeCell[cols];
			for( int col = 0; col < cols; ++col ) {
				mazeCells[row][col] = new MazeCell( row, col );
			}
		}
	}
	
	
	// Reset all the maze cells to initial state.
	// Each cell has four walls and is unreachable.
	public void initializeMazeCells() {
		for( int row = 0; row < rows; ++row ) {
			for( int col = 0; col < cols; ++col ) {
				mazeCells[row][col].initializeMazeCell();
			}
		}
	}
	
	// Get the dimensions of the maze.
	public int getRows() { return rows; }
	public int getCols() { return cols; }

	// Is row,col within the maze?
	public boolean isOnBoard( int row, int col ) {
		return (row >= 0) && (row < rows) && (col >= 0) && (col < cols);
	}
	
	
	public MazeCell getMazeCell( int row, int col ) {
		if( isOnBoard( row, col ) ) {
			return mazeCells[row][col];
		} else {
			return null;
		}
	}
	
	

	
	public class MazeCell extends PropertyListNonPersistent {
		//	Because this inherits from PropertyListNonPersistent..
		//	Each cell has its very own property list, allowing you to
		//	 attach arbitrary information to each cell.

		private int row = 0;
		private int col = 0;
		private boolean walls[] = null;

		//----------------------------------------
		//	Constructor
		//----------------------------------------
		
		MazeCell( int row, int col ) {
			this.row = row;
			this.col = col;
			
			// Initially, all walls are present,
			//  so the cell is completely enclosed.
			walls = new boolean[ 4 ];
			
			initializeMazeCell();
		}
		
		//----------------------------------------
		//	Public API
		//----------------------------------------
		
		public void initializeMazeCell() {
			// All four walls are solid.
			for( int i = FIRST_DIRECTION;  i <= LAST_DIRECTION;  ++i ) walls[i] = true;
			// Empty properties
			zapProperties();
			// Zap internal properties
			zapInternalProperties();
		}
		
		public int getRow() { return row; }
		public int getCol() { return col; }
		public String toString() { return "cell(" + row + "," + col + ")"; }
		
		
		public MazeCell getNeighboringCell( int direction ) {
			int row = neighborRow( direction );
			int col = neighborCol( direction );
			if( isOnBoard( row, col ) ) {
				return getMazeCell( row, col );
			}
			return null;
		}
		
		
		// Does this cell have one of its four walls?
		public boolean hasWall( int direction ) {
			return walls[direction];
		}

		// Set or clear a certian wall in this cell.
		public void setWall( int direction, boolean wall ) {
			walls[direction] = wall;

			// Now do the same to the corresponding wall of the neighboring cell.
			MazeCell neighbor = getNeighboringCell( direction );
			if( neighbor != null ) {
				neighbor.walls[ oppositeDirection( direction ) ] = wall;
			}
		}
		
		
		//----------------------------------------
		//	Property list access convenience -- syntax sugar
		//----------------------------------------
		
		public void setI( String propName, int value ) {
			setIntProperty( propName, value );
		}
		public int getI( String propName ) {
			return getIntProperty( propName );
		}
		
		public void setB( String propName, boolean value ) {
			setBooleanProperty( propName, value );
		}
		public boolean getB( String propName ) {
			return getBooleanProperty( propName );
		}
		

		//----------------------------------------
		//	Useful properties
		//----------------------------------------
		
		// Use this property to keep track if you have visited a cell.
		// This is useful while generating a solution.
//		private boolean hasBeenVisited = false;
//		public boolean hasBeenVisited() { return hasBeenVisited; }
//		public void setHasBeenVisited( boolean visited ) { hasBeenVisited = visited; }
		

		// Use this property to mark a cell so that it cannot be part of the solution,
		//  and cannot be part of any dead end path.
		private boolean excludedFromMaze = false;
		public boolean excludedFromMaze() { return excludedFromMaze; }
		public void setExcludedFromMaze( boolean excluded ) { excludedFromMaze = excluded; }
		

		// Is this cell on the solution path?
		private boolean isOnSolutionPath = false;
		public boolean isOnSolutionPath() { return isOnSolutionPath; }
		public void setIsOnSolutionPath( boolean isOnPath ) { isOnSolutionPath = isOnPath; }
		
		// If this cell is on the solution path, then which direction did the path enter this cell?
//		private int solutionPathEntryDirection = 0;
//		public int getSolutionPathEntryDirection() { return solutionPathEntryDirection; }
//		public void setSolutionPathEntryDirection( int direction ) { solutionPathEntryDirection = direction; }
		
		// If this cell is on the solution path, then which direction does the path exit this cell?
//		private int solutionPathExitDirection = 0;
//		public int getSolutionPathExitDirection() { return solutionPathExitDirection; }
//		public void setSolutionPathExitDirection( int direction ) { solutionPathExitDirection = direction; }

		
		// Is this cell on the dead end path?
//		private boolean isOnDeadEndPath = false;
//		public boolean isOnDeadEndPath() { return isOnDeadEndPath; }
//		public void setIsOnDeadEndPath( boolean isOnPath ) { isOnDeadEndPath = isOnPath; }
		
		// If this cell is on the dead end path, then which direction did the path enter this cell?
//		private int deadEndPathEntryDirection = 0;
//		public int getDeadEndPathEntryDirection() { return deadEndPathEntryDirection; }
//		public void setDeadEndPathEntryDirection( int direction ) { deadEndPathEntryDirection = direction; }
		
		// If this cell is on the dead end path, then which direction does the path exit this cell?
//		private int deadEndPathExitDirection = 0;
//		public int getDeadEndPathExitDirection() { return deadEndPathExitDirection; }
//		public void setDeadEndPathExitDirection( int direction ) { deadEndPathExitDirection = direction; }

		
		
		//----------------------------------------
		//	Internal support
		//----------------------------------------
		
		private void zapInternalProperties() {
//			hasBeenVisited = false;
			excludedFromMaze = false;
			
			isOnSolutionPath = false;
//			solutionPathEntryDirection = 0;
//			solutionPathExitDirection = 0;
			
//			isOnDeadEndPath = false;
//			deadEndPathEntryDirection = 0;
//			deadEndPathExitDirection = 0;
		}
		
		// Return the neighboring row if the direction is top or bottom.
		// Return this cell's row if direction is left or right.
		public int neighborRow( int direction ) {
			switch( direction ) {
				case DIRECTION_TOP:
					return row - 1;
				case DIRECTION_BOTTOM:
					return row + 1;
				default:
					return row;
			}
		}

		// Return the neighboring col if the direction is left or right.
		// Return this cell's col if the direction is up or down.
		public int neighborCol( int direction ) {
			switch( direction ) {
				case DIRECTION_LEFT:
					return col - 1;
				case DIRECTION_RIGHT:
					return col + 1;
				default:
					return col;
			}
		}

	} // class MazeCell

	
	//----------------------------------------------------------------------
    //  Internal support
    //----------------------------------------------------------------------
	
	
    //----------------------------------------------------------------------
    //  Internal support -- array of Maze Cells
    //----------------------------------------------------------------------

	private int cols;
	private int rows;
	private MazeCell[][] mazeCells = null;
		
	
} // class MazeGrid
